////////////////////////////////////////////////////////////////////////////////
//
// Copyright (c) since 1999. VASCO DATA SECURITY
// All rights reserved. http://www.vasco.com
//
////////////////////////////////////////////////////////////////////////////////

#import <Foundation/Foundation.h>
#import <MSSOrchestration/RemoteAuthenticationDisplayDataCallerDelegate.h>
#import <MSSOrchestration/OrchestrationUser.h>
NS_ASSUME_NONNULL_BEGIN

@class PasswordError;

/**
 * <p>The {@link RemoteAuthenticationDelegate} interface defines the specific elements required for the
 * integrator to interact during the Remote Authentication flow.</p>
 * <p>This flow is initiated by the server; it is started by a call to {@link OrchestrationProtocol#execute:(NSString *)}
 * with the command provided by the server.</p>
 *
 * <p>A {@link RemoteAuthenticationDelegate} object should be provided to the {@link OrchestrationProtocol} object (See {@link OrchestrationProtocol#setRemoteAuthenticationDelegate:(id<RemoteAuthenticationDelegate>)})
 * in order to receive feedback during the remote authentication process.</p>
 */
NS_REFINED_FOR_SWIFT
@protocol RemoteAuthenticationDelegate <NSObject>


/**
 * Called when the Orchestration SDK needs the {@link RemoteAuthenticationDelegate} object to display data
 * to the user for approval.
 *
 * @param dataToDisplay the data to display
 * @param caller        a {@link RemoteAuthenticationDisplayDataCallerDelegate} object to use upon user's response
 * @param user the user currently being authenticated
 */
- (void)onRemoteAuthenticationForOrchestrationUser:(OrchestrationUser *)user dataToDisplay:(NSString *)dataToDisplay caller:(id <RemoteAuthenticationDisplayDataCallerDelegate>)caller;

/**
 * Called when a step of the remote authentication process is complete. The provided orchestration
 * command is to be sent to the server.
 *
 * @param command an orchestration command that must be sent to the server
 */
- (void)onRemoteAuthenticationStepCompleteWithCommand:(NSString *)command;

/**
 * The {@link RemoteAuthenticationSuccessSessionState} lists possible session states after a Remote authentication success.
 */
NS_REFINED_FOR_SWIFT
typedef enum {
    RemoteAuthenticationSuccessSessionStateAccepted,
    RemoteAuthenticationSuccessSessionStateRefused
} RemoteAuthenticationSuccessSessionState;

/**
 * Called upon remote authentication success
 *
 * @param state the end state of the session.
 */
- (void)onRemoteAuthenticationSuccessWithState:(RemoteAuthenticationSuccessSessionState)state;

/**
  * The {@link RemoteAuthenticationSessionOutdatedReason} lists possible reasons for a session to be outdated.
  */
NS_REFINED_FOR_SWIFT
typedef enum {
    RemoteAuthenticationSessionExpired,
    RemoteAuthenticationSessionAccepted,
    RemoteAuthenticationSessionRefused,
    RemoteAuthenticationSessionUnknown
} RemoteAuthenticationSessionReason;

/**
 * Called upon remote authentication session outdated
 *
 * @param reason the reason for the session to be outdated
 */
- (void)onRemoteAuthenticationSessionOutdated:(RemoteAuthenticationSessionReason)reason;

/**
 * Called upon remote authentication abortion
 */
- (void)onRemoteAuthenticationAborted;

/**
 * Called when an error occurred during the remote authentication process
 *
 * @param error password error
 */
- (void)onRemoteAuthenticationPasswordError:(PasswordError *)error;

@end

NS_ASSUME_NONNULL_END
